#ifndef KMS_CONTROL_SYNC_CONTROL_H
#define KMS_CONTROL_SYNC_CONTROL_H

#include "global_types.h"

#include "model/attribute_list.h"

typedef enum sync_control_state_t
{
	SYNC_CO_STATE_DONE					= 0x00,
	SYNC_CO_STATE_WAITING_FOR_DEVICES	= 0x01,
	SYNC_CO_STATE_WAITING_FOR_ATTRIBUTES= 0x02,
	__SYNC_CO_LASTSTATE					= 0x04
} sync_control_state_t;

//---------------------------------- device sync functions ----------------------------------------------------------
void sync_control_init(void);

void sync_control_deinit(bool dump_missing);

int sync_control_get_pollfd(void);

error_code_t sync_control_on_event(void);

int sync_control_get_state(void);

error_code_t sync_control_check_attributes(void);
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------- timeouting ---------------------------------------------------------------------

void sync_control_activate_timeout(int timeout_ms);

int sync_control_get_remaining_timeout_time(void);
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------- filling attribute and device lists ---------------------------------------------
/**
 * Called to add a device to the list of devices we have to wait for.
 *
 * Use this function if you do not plan to set the access mode and the user and group ids of the corresponding devnode
 * of the device. The device is not added if it is already present.
 *
 * \param identifier the identifier of the device which is either a device node or a syspath
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_INVALID is returned in case the identifier is neither an absolute devnode path or an absolute syspath
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t sync_control_add_device_no_credentials(const char *identifier);

/**
 * Called to add a device to the list of devices we have to wait for.
 *
 * Use this function if you additionally want to set user and group ids as well as the access_mode of the
 * corresponding device node once it appears. If the device is already present, the user_id, group_id, and
 * access mode are set immediately.
 *
 * \param identifier the identifier of the device which is either a device node or a syspath
 * \param credentials the credentials to set
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_INVALID is returned in case the identifier is neither an absolute devnode path or an absolute syspath
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t sync_control_add_device(const char *identifier, const kms_credentials_t *credentials);

/**
 * Called to add an sysfs attribute to the list of attributes we have to wait for.
 *
 * Use this function if you do not plan to set the access mode and the user and group ids of the attribute once
 * it appears. If the attribute is already present, it is not added to the list.
 *
 * \param sysfs_path the path in the sysfs pointing to the attribute
 * \param name of the attribute
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t sync_control_add_attribute_no_credentials(const char *sysfs_path,
		const char *name);

/**
 * Called to add an sysfs attribute to the list of attributes we have to wait for.
 *
 * Use this function if you additionally want to set user and group ids as well as the access_mode of the
 * attribute once it appears. If the attribute is already present, the user_id, group_id, and access mode are
 * set immediately.
 *
 * \param sysfs_path the path in the sysfs pointing to the attribute
 * \param name of the attribute
 * \param credentials the credentials to set
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t sync_control_add_attribute(const char *sysfs_path,
		const char *name, const kms_credentials_t *credentials);


//-------------------------------------------------------------------------------------------------------------------

#endif
